<?php
/**
 * 2007-2025 Net-assembly
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 *
 * @author    Net-assembly
 * @copyright 2007-2025 Net-assembly
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License (AFL 3.0)
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/classes/CombinationValue.php';
require_once __DIR__ . '/classes/ConfigHistory.php';
require_once __DIR__ . '/classes/CarrierMatches.php';
require_once __DIR__ . '/classes/PaymentMatches.php';

class Kash extends Module
{
    protected $config_form = false;

    public function __construct()
    {
        $this->name = 'kash';
        $this->module_key = '970ad9b310f9ada921c1f80b1a4923ad';

        $this->tab = 'billing_invoicing';
        $this->version = '2.0.1';
        $this->author = 'Net-assembly';
        $this->need_instance = 1;
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Kash.click certified POS');
        $this->description = $this->l('Records Prestashop sales into a compliant POS software. Synchronize catalogue, clients, sales, etc');
        $this->ps_versions_compliancy = ['min' => '1.6', 'max' => _PS_VERSION_];
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        $sql = [];

        $sql[] = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ns_caisseconfig` (
                  `id_ns_caisseconfig` int(10) unsigned NOT NULL AUTO_INCREMENT,
                  `id_shop` INT( 11 ) UNSIGNED NOT NULL,
                  `combination_id` INT( 11 ) UNSIGNED NOT NULL,
                  `combination_value` TEXT ,
                  `encoded_value` TEXT ,
                  PRIMARY KEY (`id_ns_caisseconfig`)
                ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8';

        $sql[] = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ns_caisse_orders` (
                  `id_ns_caisse_orders` int(10) unsigned NOT NULL AUTO_INCREMENT,
                  `id_shop` INT( 11 ) UNSIGNED NOT NULL,
                  `id_order` INT( 11 ) UNSIGNED NOT NULL,
                  `url_value` TEXT ,
                  `response` INT( 11 ) UNSIGNED NOT NULL,
                  `server_message` TEXT ,
                  PRIMARY KEY (`id_ns_caisse_orders`),
                  UNIQUE  `CAISSE_ORDER_UNIQ` (`id_order`)
                ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8';

        $sql[] = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ns_transporteur` (
                  `id_ns_transporteur` int(10) unsigned NOT NULL AUTO_INCREMENT,
                  `id_shop` INT( 11 ) UNSIGNED NOT NULL,
                  `id_carrier` INT( 11 ) UNSIGNED NOT NULL,
                  `livraison_id` INT( 11 ) UNSIGNED NOT NULL,
                  PRIMARY KEY (`id_ns_transporteur`)
                ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8';

        $sql[] = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ns_paiement` (
                  `id_ns_paiement` int(10) unsigned NOT NULL AUTO_INCREMENT,
                  `id_shop` INT( 11 ) UNSIGNED NOT NULL,
                  `id_payment` INT( 11 ) UNSIGNED NOT NULL,
                  `cash_payment_id` INT( 11 ) UNSIGNED NOT NULL,
                  PRIMARY KEY (`id_ns_paiement`)
                ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8';

        $this->logData('Install');
        return parent::install() && $this->registerHook('header') && $this->registerHook('DisplayPaymentTop') && $this->registerHook('DisplayOrderConfirmation') && $this->registerHook('OrderConfirmation') && $this->runSql($sql) && $this->registerHook('actionValidateOrder');
    }

    public function uninstall()
    {
        $sql = [];

        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'ns_caisseconfig`';
        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'ns_caisse_orders`';
        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'ns_transporteur`';
        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'ns_paiement`';

        $this->runSql($sql);

        return parent::uninstall();
    }

    public function runSql($sql)
    {
        foreach ($sql as $s) {
            if (!Db::getInstance()->Execute($s)) {
                return false;
            }
        }
        return true;
    }

    public function getContent()
    {
        $homeWS = 'http://php7.enregistreuse.fr';
        $homeWSi ='https://kash.click';
        if (Context::getContext()->shop->getBaseURL(true) == 'http://prestashop/') {
            $homeWS = 'http://caisses';
            $homeWSi = 'http://caisses';
        }

        $this->context->smarty->assign('token', Tools::getAdminTokenLite('AdminModules'));
        $this->context->smarty->assign('id_current_shop', $this->context->shop->id);
        $this->context->smarty->assign('current_shop', $this->context->shop);

        $id_lang = Configuration::get('PS_LANG_DEFAULT');
        $shop_list = Shop::getShops(true);
        $this->context->smarty->assign('shop_list', $shop_list);

        if (Tools::isSubmit('viewLogin') || Tools::isSubmit('submitLogin')) {
            $accountKEYAPI = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_API', null, null, $this->context->shop->id);
            $shopID = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', null, null, $this->context->shop->id);
            $idck = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_IDCK', null, null, $this->context->shop->id);

            $MYMODULE_WS_KEY = Configuration::get('MYMODULE_WS_KEY', null, null, $this->context->shop->id);

            if (!$MYMODULE_WS_KEY) {
                try {
                    if (function_exists('random_bytes')) {
                        $key = bin2hex(random_bytes(16));
                    } else {
                        $key = strtoupper(Tools::passwdGen(32));
                    }
                } catch (\Exception $e) {
                    $key = md5(uniqid((string) time(), true));
                }

                $ws = new \WebserviceKey();
                $ws->key = $key;
                $ws->active = true;
                $ws->description = $this->l('free-cash-register.net');

                if ($ws->add()) {
                    $idWs = (int) $ws->id;
    
                    $resources = [];
                    if (class_exists('\WebserviceRequest') && method_exists('\WebserviceRequest', 'getResources')) {
                        $resources = array_keys(\WebserviceRequest::getResources());
                    }
    
                    if (empty($resources)) {
                        $resources = [
                            'addresses',
                            'carriers',
                            'cart_rules',
                            'carts',
                            'categories',
                            'combinations',
                            'configurations',
                            'contacts',
                            'countries',
                            'currencies',
                            'customers',
                            'customer_threads',
                            'deliveries',
                            'employees',
                            'images',
                            'image_types',
                            'manufacturers',
                            'messages',
                            'order_carriers',
                            'order_details',
                            'order_histories',
                            'order_invoices',
                            'order_payments',
                            'order_slip',
                            'order_states',
                            'orders',
                            'price_ranges',
                            'products',
                            'shops',
                            'shop_groups',
                            'specific_prices',
                            'states',
                            'stock_availables',
                            'stock_movements',
                            'stocks',
                            'suppliers',
                            'tags',
                            'taxes',
                            'tax_rule_groups',
                            'tax_rules',
                            'translated_configurations',
                            'warehouse_product_locations',
                            'warehouses',
                            'weight_ranges',
                            'zones',
                        ];
                    }
    
                    $verbs = ['GET'];
    
                    \Db::getInstance()->execute('DELETE FROM ' . _DB_PREFIX_ . 'webservice_permission WHERE id_webservice_account=' . (int) $idWs);
    
                    $rows = [];
                    foreach ($resources as $res) {
                        foreach ($verbs as $verb) {
                            $rows[] = [
                                'id_webservice_account' => $idWs,
                                'resource' => pSQL($res),
                                'method' => pSQL($verb),
                            ];
                        }
                    }
    
                    \Db::getInstance()->insert('webservice_permission', $rows);
    
                    \Configuration::updateValue('MYMODULE_WS_KEY', $key);
    
                    $MYMODULE_WS_KEY = $key;
                    Configuration::updateValue('PS_WEBSERVICE', 1);
    
                    $showMsgs[] = $this->l('Webservice key created for free-cash-register.net');
                }
            }

            if (Tools::isSubmit('submitLogin')) {
                $login = Tools::getValue('log_login');
                $pw = Tools::getValue('log_pw');

                $url = $homeWS . '/workers/getAuthToken.php';

                $data = [
                    'login' => $login,
                    'password' => $pw,
                    'moduleKey' => $MYMODULE_WS_KEY,
                    'prestaHome' => Context::getContext()->shop->getBaseURL(true),
                ];

                $options = [
                    'http' => [
                        'method' => 'POST',
                        'header' => "Content-Type: application/x-www-form-urlencoded\r\n",
                        'content' => http_build_query($data),
                    ],
                ];

                $context = stream_context_create($options);
                $result = file_get_contents($url, false, $context);

                $resJson = json_decode($result, true);
                if ($resJson && $resJson['result'] == 'OK') {
                    Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_API', $resJson['APIKEY'], false, null, $this->context->shop->id);
                    Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', $resJson['SHOPID'], false, null, $this->context->shop->id);
                    Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_IDCK', $resJson['COOKIE'], false, null, $this->context->shop->id);

                    $accountKEYAPI = $resJson['APIKEY'];
                    $shopID = $resJson['SHOPID'];
                    $idck = $resJson['COOKIE'];

                    $showMsgs[] = $this->l('Login succeeded');
                } elseif ($resJson['errorMessage']) {
                    $showMsgs[] = $resJson['errorMessage'];
                } elseif ($resJson['result']) {
                    $showMsgs[] = $resJson['result'];
                }
            }

            if (Tools::isSubmit('doLogout')) {
                Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_API', '', false, null, $this->context->shop->id);
                Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', '', false, null, $this->context->shop->id);
                Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_IDCK', '', false, null, $this->context->shop->id);
                $accountKEYAPI = '';
                $shopID = '';
                $idck = '';
            }

            if (Tools::isSubmit('createAccount')) {
                $urlCr = $homeWSi . '/workers/updaters/boutique_add.php?nomBoutique=' . urlencode(Configuration::get('PS_SHOP_NAME')) . '&email=' . Tools::getValue('create_email') . '&prestaShopURL=' . urlencode(Context::getContext()->shop->getBaseURL(true)) . '&prestaShopApiKey=' . $MYMODULE_WS_KEY;
                $result = file_get_contents($urlCr);

                $resJson = json_decode($result, true);
                if ($resJson && $resJson['result'] == 'OK') {
                    Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_API', $resJson['APIKEY'], false, null, $this->context->shop->id);
                    Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', $resJson['SHOPID'], false, null, $this->context->shop->id);
                    Configuration::updateValue('CAISSEENREGISTREUSE_ACCOUNT_IDCK', $resJson['COOKIE'], false, null, $this->context->shop->id);

                    $accountKEYAPI = $resJson['APIKEY'];
                    $shopID = $resJson['SHOPID'];
                    $idck = $resJson['COOKIE'];

                    $showMsgs[] = $this->l('Account creation succeeded');
                } elseif ($resJson['result']) {
                    $showMsgs[] = $resJson['result'];
                } else {
                    $showMsgs[] = $result;
                }
            }
        }

        $output = $this->addMenu();

        if ((bool) Tools::isSubmit('submitCaisseEnregistreuseModule')) {
            $this->postProcess();
        }

        if (Tools::isSubmit('viewHelp')) {
            $this->context->smarty->assign('module_dir', $this->_path);
            $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/configure.tpl');
        } elseif (Tools::isSubmit('viewCombinations') || Tools::isSubmit('submitAddCombination') || Tools::isSubmit('updatens_caisseconfig') || Tools::isSubmit('deletens_caisseconfig')) {
            $shopID = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', null, null, $this->context->shop->id);
            $accountKEYAPI = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_API', null, null, $this->context->shop->id);
            $url = 'http://caisse.enregistreuse.fr/workers/getDeclinaisons.php?idboutique=' . $shopID . '&key=' . $accountKEYAPI . '&format=json';
            $dataDecl = file_get_contents($url);
            $jsonD = json_decode($dataDecl, true);
            $rr = [];
            foreach ($jsonD as $je) {
                $rr[] = ['value' => $je['id'], 'label' => $je['nom']];
            }
            $this->context->smarty->assign('tonmodule_options', $rr);

            $idLang = (int) $this->context->language->id;
            $idShop = (int) $this->context->shop->id;

            $selectedAttrId = 0;

            $prefix = _DB_PREFIX_;
            $sql = '
                SELECT a.id_attribute, a.id_attribute_group,
                       al.name AS attribute_name,
                       agl.name AS group_name
                FROM ' . $prefix . 'attribute a
                INNER JOIN ' . $prefix . 'attribute_lang al
                    ON (al.id_attribute = a.id_attribute AND al.id_lang = ' . $idLang . ')
                INNER JOIN ' . $prefix . 'attribute_group ag
                    ON (ag.id_attribute_group = a.id_attribute_group)
                INNER JOIN ' . $prefix . 'attribute_group_lang agl
                    ON (agl.id_attribute_group = a.id_attribute_group AND agl.id_lang = ' . $idLang . ')
                LEFT JOIN ' . $prefix . 'attribute_shop as_ ON (as_.id_attribute = a.id_attribute AND as_.id_shop = ' . $idShop . ')
                LEFT JOIN ' . $prefix . 'attribute_group_shop ags ON (ags.id_attribute_group = ag.id_attribute_group AND ags.id_shop = ' . $idShop . ')
                ORDER BY agl.name ASC, a.position ASC, al.name ASC
            ';
            $rows = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);

            $options = [];
            foreach ($rows as $r) {
                $options[] = [
                    'value' => (string) $r['id_attribute'],
                    'label' => sprintf('%s: %s', $r['group_name'], $r['attribute_name']),
                    'group' => $r['group_name'],
                ];
            }

            $this->context->smarty->assign([
                'attr_options' => $options,
                'attr_selected' => $selectedAttrId,
                'attr_fieldname' => 'combination_id',
            ]);

            if (Tools::getValue('id_ns_caisseconfig') != '' && Tools::isSubmit('deletens_caisseconfig')) {
                $id_ns_caisseconfig = (int) Tools::getValue('id_ns_caisseconfig');
                $sampleObj = new CombinationValue($id_ns_caisseconfig);
                $sampleObj->delete();
                $output .= $this->displayConfirmation($this->l('Item  Removed'));
            }

            if (Tools::getValue('id_ns_caisseconfig') != '' && Tools::isSubmit('updatens_caisseconfig')) {
                $id_ns_caisseconfig = (int) Tools::getValue('id_ns_caisseconfig');
                $sampleObj = new CombinationValue($id_ns_caisseconfig);
                $this->context->smarty->assign('sampleObj', $sampleObj);
            }

            if (Tools::isSubmit('submitAddCombination')) {
                $output .= $this->addCombination();
            }

            $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/config_combinations.tpl');
            $output .= $this->renderItemsComb();
        } elseif (Tools::isSubmit('viewHistory') || Tools::isSubmit('deletens_caisse_orders')) {
            $id_shop = $this->context->shop->id;

            if (Tools::isSubmit('deletens_caisse_orders') && Tools::getValue('id_ns_caisse_orders') != '') {
                $id_ns_caisse_orders = (int) Tools::getValue('id_ns_caisse_orders');
                $ConfigHistory = new ConfigHistory($id_ns_caisse_orders);
                $ConfigHistory->delete();

                $output .= $this->displayConfirmation($this->l('Item  Removed'));
            }
            $all_order_histories = ConfigHistory::getHistories($id_shop);
            $this->context->smarty->assign('all_order_histories', $all_order_histories);

            $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/config_history.tpl');
        } elseif (Tools::isSubmit('viewOptions') || Tools::isSubmit('submitConfigOptions')) {
            if (Tools::isSubmit('submitConfigOptions')) {
                $output .= $this->updateOptionValues();
            }
            $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/config_options.tpl');
        } elseif (Tools::isSubmit('viewCarrierMatches') || Tools::isSubmit('submitCarrierMatches') || Tools::isSubmit('deletens_transporteur')) {
            $this->context->smarty->assign('carrier_list', Carrier::getCarriers((int) $id_lang, true));

            if (Tools::isSubmit('submitCarrierMatches')) {
                $id_carrier = Tools::getValue('id_carrier');
                $livraison_id = Tools::getValue('livraison_id');
                $output .= $this->addNewMatches($id_carrier, $livraison_id);
            }

            if (Tools::isSubmit('deletens_transporteur') && Tools::getValue('id_ns_transporteur')) {
                $id_ns_transporteur = Tools::getValue('id_ns_transporteur');
                $TransPorteur = new CarrierMatches($id_ns_transporteur);
                $TransPorteur->delete();
                $output .= $this->displayConfirmation($this->l('Item  Removed'));
            }

            $this->context->smarty->assign('carrier_matches_list', CarrierMatches::getMatches($this->context->shop->id));
            $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/config_carriers.tpl');
        } elseif (Tools::isSubmit('viewPaymentMatches') || Tools::isSubmit('submitPaymentMatches') || Tools::isSubmit('deletens_paiement')) {
            $shopID = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', null, null, $this->context->shop->id);
            $accountKEYAPI = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_API', null, null, $this->context->shop->id);

            $url = $homeWS . '/workers/getPaymentModes.php?idboutique=' . $shopID . '&key=' . $accountKEYAPI . '&format=json';
            $dataDecl = file_get_contents($url);
            $jsonD = json_decode($dataDecl, true);
            $rr = [
                ['value' => -1, 'label' => $this->l('Not paid, Validated')],
                ['value' => -2, 'label' => $this->l('Not paid, not validated')],
            ];
            foreach ($jsonD as $je) {
                $rr[] = ['value' => $je['id'], 'label' => $je['nomlong']];
            }
            $this->context->smarty->assign('tonmodule_options', $rr);

            $this->context->smarty->assign('payment_methods', $this->getPaymentMethods());

            if (Tools::isSubmit('deletens_paiement') && Tools::getValue('id_ns_paiement')) {
                $id_ns_paiement = Tools::getValue('id_ns_paiement');
                $PaymentMatches = new PaymentMatches($id_ns_paiement);
                $PaymentMatches->delete();
                $output .= $this->displayConfirmation($this->l('Item  Removed'));
            }

            if (Tools::isSubmit('submitPaymentMatches')) {
                $id_payment = Tools::getValue('id_payment');
                $cash_payment_id = Tools::getValue('cash_payment_id');
                $output .= $this->addNewPaymentMatches($id_payment, $cash_payment_id);
            }
            $this->context->smarty->assign('payment_matches_list', PaymentMatches::getMatches($this->context->shop->id));
            $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/config_payments.tpl');
        } elseif (Tools::isSubmit('viewSync') || Tools::isSubmit('dosync')) {
            $idck = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_IDCK', null, null, $this->context->shop->id);
            if (Tools::isSubmit('dosync')) {
                $subURL = $homeWS . '/workers/forms/systeme/PLUimportResult.php?utilisateur_id_ck=' . $idck . '&prestaShopApiKey=Auto&customForm=1&execute=1';
                if (Tools::getValue('applyADD')) {
                    $subURL .= '&applyADD=1';
                }
                if (Tools::getValue('applyDEL')) {
                    $subURL .= '&applyDEL=1';
                }
                if (Tools::getValue('applyUPD')) {
                    $subURL .= '&applyUPD=1';
                }

                $content = Tools::file_get_contents($subURL);

                $this->context->smarty->assign([
                    'remote_content' => $content,
                ]);

                $output .= $this->context->smarty->fetch(
                    $this->local_path . 'views/templates/admin/remote_content.tpl'
                );
            } else {
                $f = file_get_contents($homeWS . '/workers/forms/systeme/PLUimportResult.php?utilisateur_id_ck=' . $idck . '&prestaShopApiKey=Auto&customForm=1');
                if (!$f) {
                    $output .= $this->displayConfirmation('Failed getting ' . $homeWS . '/workers/forms/systeme/PLUimportResult.php?utilisateur_id_ck=' . $idck . '&prestaShopApiKey=Auto&customForm=1');
                } else {
                    $this->context->smarty->assign('syncContent', $f);
                    $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/validateSync.tpl');
                }
            }
        } elseif (Tools::isSubmit('viewLogin') || Tools::isSubmit('submitLogin')) {
            if (isset($showMsgs)) {
                foreach ($showMsgs as $m) {
                    $output .= $this->displayConfirmation($m);
                }
            }
            if (!isset($accountKEYAPI) || !$accountKEYAPI) {
                $this->context->smarty->assign('busEmail', Configuration::get('PS_SHOP_EMAIL'));
                $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/dologin.tpl');
                $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/createAccount.tpl');
            } else {
                $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/doLogout.tpl');
            }
        } elseif (Tools::isSubmit('viewCaisse')) {
            $output .= $this->displayCaisse();
        } else {
            $output .= $this->context->smarty->fetch($this->local_path . 'views/templates/admin/config_home.tpl');
        }

        return $output;
    }

    public function addMenu()
    {
        $accountKEYAPI = (string) Configuration::get(
            'CAISSEENREGISTREUSE_ACCOUNT_API',
            null,
            null,
            (int) $this->context->shop->id
        );

        $adminLinkBase = $this->context->link->getAdminLink('AdminModules', false);
        $token = Tools::getAdminTokenLite('AdminModules');

        $this->context->smarty->assign([
            'admin_link_base' => $adminLinkBase,
            'token' => $token,
            'module_name' => $this->name,
            'has_account_api' => !empty($accountKEYAPI),
            'label_principle' => $this->l('Principle'),
            'label_account_connection' => $this->l('Account connection'),
            'label_synchronize' => $this->l('Synchronize catalogue data'),
            'label_order_history' => $this->l('Order History'),
            'label_options' => $this->l('Options'),
            'label_carriers' => $this->l('Carriers'),
            'label_payments' => $this->l('Payments'),
            'label_start_pos' => $this->l('Start POS'),
            'label_about' => $this->l('About'),
        ]);

        return $this->context->smarty->fetch($this->local_path . 'views/templates/admin/menu.tpl');
    }

    public function displayCaisse()
    {
        $idck = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_IDCK', null, null, $this->context->shop->id);

        $homeWSi = $this->l('https://www.free-cash-register.net');

        if (Context::getContext()->shop->getBaseURL(true) == 'http://prestashop/') {
            $homeWSi = 'http://caisses';
        }
        $urlcaisse = $homeWSi . '?utilisateur_id_ck=' . $idck;

        $html = '<iframe src="' . $urlcaisse . '" width="100%" height="700px;" border="0" style="border:none;"></iframe>';

        return $html;
    }

    public function addCombination()
    {
        $message = '';

        $is_duplicated = CombinationValue::verifyByNUmber((int) Tools::getValue('combination_id'));

        if (Tools::isSubmit('submitAddCombination') && Tools::getValue('combination_value') != '' && Tools::getValue('combination_id') != '') {
            $id_ns_caisseconfig = Tools::getValue('id_ns_caisseconfig');

            $sampleObj = CombinationValue::loadByIdCombination($id_ns_caisseconfig);
            if ($sampleObj !== null) {
                $encoded_value = Tools::getValue('combination_value');
                $sampleObj->encoded_value = rawurlencode(strtoupper(str_replace(' ', '', $encoded_value)));
                $sampleObj->combination_value = Tools::getValue('combination_value');

                $sampleObj->combination_id = Tools::getValue('combination_id');
                $sampleObj->id_shop = (int) Tools::getValue('id_shop');

                if (isset($sampleObj->id) && Tools::getValue('id_ns_caisseconfig') != '') {
                    $sampleObj->update();
                    $message = $this->displayConfirmation($this->l('Item updated'));
                } else {
                    $sampleObj->add();
                    $message = $this->displayConfirmation($this->l('Item  added'));
                }
            }
        }
        return $message;
    }

    public function renderItemsComb()
    {
        $fields_list = [
            'id_ns_caisseconfig' => [
                'title' => $this->l('ID'),
                'search' => false,
            ],
            'id_shop' => [
                'title' => $this->l('ID SHOP'),
                'search' => false,
            ],
            'combination_value' => [
                'title' => $this->l('Combination'),
                'search' => false,
            ],
            'encoded_value' => [
                'title' => $this->l('Encoded'),
                'search' => false,
            ],
            'combination_id' => [
                'title' => $this->l('VALUE'),
                'search' => false,
            ],
        ];

        $helper_list = new HelperList();
        $helper_list->module = $this;
        $helper_list->title = $this->l('Combination: Items');
        $helper_list->shopLinkType = '';
        $helper_list->no_link = true;
        $helper_list->show_toolbar = true;
        $helper_list->simple_header = false;
        $helper_list->identifier = 'id_ns_caisseconfig';
        $helper_list->table = 'ns_caisseconfig';
        $helper_list->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&viewCombinations&configure=' . $this->name;
        $helper_list->token = Tools::getAdminTokenLite('AdminModules');
        $helper_list->actions = ['edit', 'delete'];
        $helper_list->bulk_actions = [
            'select' => [
                'text' => $this->l('Change order status'),
                'icon' => 'icon-refresh',
            ],
        ];

        $id_shop = (int) $this->context->shop->id;
        $orders = CombinationValue::getItems($id_shop);
        $helper_list->listTotal = count($orders);

        $page = Tools::getValue('submitFilter' . $helper_list->table);
        if (!$page) {
            $page = 1;
        }
        $pagination = Tools::getValue($helper_list->table . '_pagination');
        if (!$pagination) {
            $pagination = 50;
        }
        $orders = $this->paginateOrderProducts($orders, $page, $pagination);

        return $helper_list->generateList($orders, $fields_list);
    }

    public function paginateOrderProducts($orders, $page = 1, $pagination = 50)
    {
        if (count($orders) > $pagination) {
            $orders = array_slice($orders, $pagination * ($page - 1), $pagination);
        }
        return $orders;
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitCaisseEnregistreuseModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
        . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => $this->getConfigFormValues(), /* Add values for your inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        ];

        return $helper->generateForm([$this->getConfigForm()]);
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm()
    {
        return [
            'form' => [
                'legend' => [
                    'title' => 'old',
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'col' => 3,
                        'type' => 'text',
                        'desc' => $this->l('Enter your login'),
                        'name' => 'CAISSEENREGISTREUSE_LOGIN',
                        'label' => $this->l('Login'),
                    ],
                    [
                        'col' => 3,
                        'type' => 'password',
                        'desc' => $this->l('Enter your password'),
                        'name' => 'CAISSEENREGISTREUSE_PW',
                        'label' => 'Password',
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];
    }

    /**
     * Set values for the inputs.
     */
    protected function alphanum($tmp)
    {
        $res = '';
        $str = $tmp;
        for ($i = 0; $i <= strlen($str); ++$i) {
            if ((($str[$i] >= 'a' && $str[$i] <= 'z') || ($str[$i] >= 'A' && $str[$i] <= 'Z') || ($str[$i] >= '0' && $str[$i] <= '9')) || $str[$i] == '.' || $str[$i] == '_' || $str[$i] == '-') {
                $res .= $str[$i];
            }
        }
        return $res;
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        return [
            'CAISSEENREGISTREUSE_ACCOUNT_API' => Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_API', null, null, $this->context->shop->id),
            'CAISSEENREGISTREUSE_ACCOUNT_SHOPID' => Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', null, null, $this->context->shop->id),
        ];
    }

    protected function logData($txt)
    {
        $f = fopen($this->local_path . 'logBackground.txt', 'a+');
        fwrite($f, date('m/d/Y H:i ') . $txt . "\r\n");
        fclose($f);
    }

    /**
     * Save form data.
     */
    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();
        foreach (array_keys($form_values) as $key) {
            Configuration::updateValue($key, Tools::getValue($key), false, null, $this->context->shop->id);
        }
    }

    public function hookBackOfficeHeader()
    {
    }

    public function hookHeader()
    {
        $this->context->controller->addJS($this->_path . '/views/js/front.js');
        $this->context->controller->addCSS($this->_path . '/views/css/front.css');
    }

    public function hookActionCustomerAccountUpdate($params)
    {
    }
    public function hookActionUpdateQuantity($params)
    {
    }
    public function hookActionCustomerAccountAdd($params)
    {
    }
    public function hookActionPaymentConfirmation($params)
    {
    }
    public function hookActionOrderHistoryAddAfter($params)
    {
    }

    public function hookActionValidateOrder($params)
    {
        return '';
    }
    public function hookDisplayPayment($params)
    {
    }

    public function hookDisplayPaymentTop($params)
    {
    }

    public function hookDisplayPaymentReturn()
    {
    }

    public function hookDisplayOrderConfirmation($params)
    {
        $id_order = (int) Tools::getValue('id_order');

        if (Configuration::get('SEND_ORDERS', null, null, $this->context->shop->id) == 1) {
            if (ConfigHistory::isOrderExist($id_order)) {
                return '';
            } else {
                return $this->hookOrderConfirmation($params);
            }
        }
    }

    public function hookOrderConfirmation($params)
    {
        $accountKEYAPI = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_API', null, null, $this->context->shop->id);
        $shopID = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', null, null, $this->context->shop->id);
        $liveMode = true;
        $debugMode = false;

        if ($accountKEYAPI == null || $accountKEYAPI == 'xxxxxxxx' || $accountKEYAPI == '') {
            return;
        }

        $id_order = (int) Tools::getValue('id_order');
        $order = new Order($id_order);

        $nka_current_state = $order->current_state;
        $status = new OrderState((int) $nka_current_state);

        $dateValeur = $this->getDateLivraison($params);

        $data = print_r($params, true);

        $tran3 = $this->getTransporteur($params);

        $codeBarreClient = 'Pr' . $this->context->customer->id;
        $modePaiement = $params['order']->payment;
        $paiement1 = $this->getPaymentMethods();
        $id_payment = (int) Tools::getValue('id_module');
        $paiement3 = PaymentMatches::getPaymentMethod($id_payment);

        $customer = $this->context->customer;
        $addedItems = $this->getAddedItems($params);
        $infosClient = $this->getInfosClient($params);
        $message = $params['order']->gift_message;
        $messager = rawurlencode($message);

        $homeWS = 'http://caisse.enregistreuse.fr';
        if (Context::getContext()->shop->getBaseURL(true) == 'http://prestashop/') {
            $homeWS = 'http://caisses';
        }
        $urlWS = $homeWS . '/workers/webapp.php?idboutique=' . $shopID . '&key=' . $accountKEYAPI . '&payment=' . $paiement3;
        $urlWS .= '&dateValeur=' . $dateValeur . '&idUser=Prestashop' . $infosClient . '&deliveryMethod=' . $tran3 . '&comment=' . $messager;

        $options = [
            'http' => [
                'method' => 'POST',
                'header' => "Content-Type: application/x-www-form-urlencoded\r\n" .
                "Accept: application/json\r\n",
                'content' => $addedItems,
                'timeout' => 30,
            ],
        ];

        $context = stream_context_create($options);

        $result = file_get_contents($urlWS, false, $context);

        if ($result === false) {
            $error = error_get_last();
        }

        $recordResult = json_decode(str_replace('\'', '"', $result));

        $response = null;
        $server_message = null;

        if ($recordResult) {
            if (property_exists($recordResult, 'errorMessage')) {
                $response = 0;
                $server_message = 'Server returned explicit error message : ' . $recordResult->errorMessage;
            }
            if (property_exists($recordResult, 'orderID')) {
                $response = 1;
                $server_message = 'Order was successfully created : ' . $recordResult->orderID;
            }
        }

        $this->sendHistory($params, $response, $server_message, $urlWS . $addedItems);

        $this->sendEmail($urlWS . $addedItems, $server_message, $params);
    }

    public function sendHistory($params, $response, $server_message, $urlWS)
    {
        $id_customer = (int) ($this->context->customer->logged ? $this->context->customer->id : false);

        $shopID = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_SHOPID', null, null, $this->context->shop->id);
        $accountKEYAPI = Configuration::get('CAISSEENREGISTREUSE_ACCOUNT_API', null, null, $this->context->shop->id);

        $id_order = (int) Tools::getValue('id_order');
        $order = new Order($id_order);
        $nka_current_state = $order->current_state;
        $status = new OrderState((int) $nka_current_state);

        $ConfigHistory = ConfigHistory::loadByIdOrder($id_order);
        $ConfigHistory->id_order = $params['order']->id;
        $ConfigHistory->id_shop = $this->context->shop->id;
        $ConfigHistory->url_value = $urlWS;
        $ConfigHistory->response = $response;
        $ConfigHistory->server_message = $server_message;

        if (isset($ConfigHistory->id)) {
            $ConfigHistory->update();
        } else {
            $ConfigHistory->add();
        }
    }

    public function getInfosClient($params)
    {
        $customer = $this->context->customer;
        $id_address_first = Address::getFirstCustomerAddressId($customer->id, true);
        $id_address_delivery = $params['order']->id_address_delivery;
        $Address = new Address($id_address_delivery);

        $infosClient = '';
        $infosClient .= '&client[nom]=' . rawurlencode($customer->lastname);
        $infosClient .= '&client[prenom]=' . rawurlencode($customer->firstname);
        $infosClient .= '&client[email]=' . rawurlencode($customer->email);
        $infosClient .= '&client[telephone]=' . rawurlencode($Address->phone);
        $infosClient .= '&client[adresseligne1]=' . rawurlencode($Address->address1);
        $infosClient .= '&client[adresseligne2]=' . rawurlencode($Address->address2);
        $infosClient .= '&client[codepostal]=' . rawurlencode($Address->postcode);
        $infosClient .= '&client[ville]=' . rawurlencode($Address->city);
        $infosClient .= '&client[pays]=' . rawurlencode($Address->country);
        $infosClient .= '&client[rcs]=' . rawurlencode($customer->siret);
        $infosClient .= '&client[codeBarre]=Pr' . ($customer->id + 30000);
        return $infosClient;
    }

    public function getDeclinaison($params)
    {
        $id_order = (int) Tools::getValue('id_order');
        $order = new Order($id_order);
        $id_cart = (int) $order->id_cart;
        $cart = new Cart($id_cart);
        $cart_products = $cart->getProducts();

        $decli4 = '';

        foreach ($cart_products as $product) {
            $decli = rawurlencode($product['attributes']);
            $liste = CombinationValue::doReplace($decli);
            $liste1 = explode('-', $liste);
            $nompizza = $liste1[0];
            $decli4 = $nompizza;
        }

        return $decli4;
    }

    public function getAddedItems($params)
    {
        $id_order = (int) Tools::getValue('id_order');
        $order = new Order($id_order);

        $productList = $order->getProducts();

        $addedItems = '';

        foreach ($productList as $product) {
            $attrsStructured = [];
            if (!empty($product['product_attribute_id'])) {
                $combination = new Combination((int) $product['product_attribute_id']);
                if (Validate::isLoadedObject($combination)) {
                    $names = $combination->getAttributesName((int) $this->context->language->id);
                    if (is_array($names)) {
                        foreach ($names as $attr) {
                            $attrsStructured[] = $this->alphanum($attr['name']);
                        }
                    }
                }
            }
            $addedItems .= '&itemsList[]=|Pr' . $product['product_id'] . '_' . $product['product_quantity'] . '__' . rawurlencode('' . ($product['total_wt'] / $product['product_quantity'])) . '_' . implode('_', $attrsStructured);
        }

        $totalDiscount = $params['order']->total_discounts;
        if ((float) $totalDiscount != 0) {
            $addedItems .= '&itemsList[]=-1_1_' . rawurlencode('Discount') . '_-' . $totalDiscount;
        }

        $shippingPrice = $params['order']->total_shipping;
        if ((float) $shippingPrice != 0) {
            $addedItems .= '&itemsList[]=-0_1_' . rawurlencode('Shipping') . '_' . $shippingPrice;
        }

        return $addedItems;
    }

    public function getTransporteur($params)
    {
        $id_carrier = $params['order']->id_carrier;
        $carrier_matche = CarrierMatches::getDeliveryMethod($id_carrier);

        return $carrier_matche;
    }

    public function getDateLivraison($params)
    {
        $dat = $params['cookie']->msg;
        $d = $params['order']->delivery_date;
        $d1 = strtotime($dat);
        $time_difference = Configuration::get('TIME_DIFFERENCE', null, null, $this->context->shop->id);
        $dateValeur = strtotime($params['order']->delivery_date) - ($time_difference * 3600);
        $dateValeur = str_replace('-62169983636', '', '' . $dateValeur);

        return $dateValeur;
    }

    public function updateOptionValues()
    {
        $message = '';

        if (Tools::isSubmit('submitConfigOptions')) {
            Configuration::updateValue('TIME_DIFFERENCE', (int) Tools::getValue('TIME_DIFFERENCE'), false, null, $this->context->shop->id);
            Configuration::updateValue('EMAIL_ADRESS', Tools::getValue('EMAIL_ADRESS'), false, null, $this->context->shop->id);
            Configuration::updateValue('RECEIVE_EMAIL_NOTIFICATION', (bool) Tools::getValue('RECEIVE_EMAIL_NOTIFICATION'), false, null, $this->context->shop->id);
            Configuration::updateValue('SEND_ORDERS', (bool) Tools::getValue('SEND_ORDERS'), false, null, $this->context->shop->id);

            Configuration::updateValue('USER_PrestaShop_CODE', (int) Tools::getValue('USER_PrestaShop_CODE'), false, null, $this->context->shop->id);
            Configuration::updateValue('USER_Lucien_CODE', (int) Tools::getValue('USER_Lucien_CODE'), false, null, $this->context->shop->id);

            $message = $this->displayConfirmation($this->l('Options  Saved  Successfully !'));
        }
        return $message;
    }

    public function sendEmail($url, $server_message, $params)
    {
        if (Configuration::get('RECEIVE_EMAIL_NOTIFICATION', null, null, $this->context->shop->id) == 1 && Configuration::get('EMAIL_ADRESS', null, null, $this->context->shop->id) != '') {
            $email = Configuration::get('EMAIL_ADRESS', null, null, $this->context->shop->id);
            $to = $email;

            $message_title = $this->l('Notification about new Order');

            $order = $params['order'];
            $nka_current_state = $order->current_state;
            $order_state = new OrderState((int) $nka_current_state);

            $context = Context::getContext();
            $id_lang = (int) $context->language->id;
            $id_shop = (int) $context->shop->id;
            $currency = $this->context->currency;
            $customer = $this->context->customer;
            $configuration = Configuration::getMultiple(
                [
                    'PS_SHOP_EMAIL',
                    'PS_MAIL_METHOD',
                    'PS_MAIL_SERVER',
                    'PS_MAIL_USER',
                    'PS_MAIL_PASSWD',
                    'PS_SHOP_NAME',
                    'PS_MAIL_COLOR',
                ], $id_lang, null, $id_shop
            );

            $delivery = new Address((int) $order->id_address_delivery);
            $invoice = new Address((int) $order->id_address_invoice);
            $order_date_text = Tools::displayDate($order->date_add);
            $carrier = new Carrier((int) $order->id_carrier);
            $message = $this->getAllMessages($order->id);

            if (!$message) {
                $message = $this->l('No message');
            }

            $products = $params['order']->getProducts();
            $customized_datas = Product::getAllCustomizedDatas((int) $params['cart']->id);
            Product::addCustomizationPrice($products, $customized_datas);

            foreach ($products as $key => $product) {
                $unit_price = Product::getTaxCalculationMethod($customer->id) == PS_TAX_EXC ? $product['product_price'] : $product['product_price_wt'];

                $customization_text = '';
                if (isset($customized_datas[$product['product_id']][$product['product_attribute_id']])) {
                    foreach ($customized_datas[$product['product_id']][$product['product_attribute_id']][$order->id_address_delivery] as $customization) {
                        if (isset($customization['datas'][Product::CUSTOMIZE_TEXTFIELD])) {
                            foreach ($customization['datas'][Product::CUSTOMIZE_TEXTFIELD] as $text) {
                                $customization_text .= $text['name'] . ': ' . $text['value'] . '<br />';
                            }
                        }

                        if (isset($customization['datas'][Product::CUSTOMIZE_FILE])) {
                            $customization_text .= count($customization['datas'][Product::CUSTOMIZE_FILE]) . ' ' . $this->l('image(s)') . '<br />';
                        }

                        $customization_text .= '---<br />';
                    }
                    if (method_exists('Tools', 'rtrimString')) {
                        $customization_text = Tools::rtrimString($customization_text, '---<br />');
                    } else {
                        $customization_text = preg_replace('/---<br \/>$/', '', $customization_text);
                    }
                }

                $url = $context->link->getProductLink($product['product_id']);
            }
            if ($delivery->id_state) {
                $delivery_state = new State((int) $delivery->id_state);
            }
            if ($invoice->id_state) {
                $invoice_state = new State((int) $invoice->id_state);
            }

            if (Product::getTaxCalculationMethod($customer->id) == PS_TAX_EXC) {
                $total_products = $order->getTotalProductsWithoutTaxes();
            } else {
                $total_products = $order->getTotalProductsWithTaxes();
            }

            $message_title = $this->l('Notification about new Order');
            $message_subject = $this->l('Hi, new order was sent to the cash  system.');
            $notif_det_text = $this->l('Notification details');
            $notif_det_date = $this->l('Sent on:');

            $date = date('d/m/y');
            $base_dir_nka = Tools::getHttpHost(true) . __PS_BASE_URI__;

            $html = '';
            $html .= $server_message;
            $html .= '<br/> URL: ' . $url;

            return Mail::Send(
                $this->context->language->id,
                'new_order',
                Mail::l($message_title, $this->context->language->id),
                [
                    '{html}' => $html,
                    '{date}' => $date,
                    '{base_dir_nka}' => $base_dir_nka,
                    '{message_title}' => $message_title,
                    '{message_subject}' => $message_subject,
                    '{username}' => '',
                    '{phone}' => '',
                    '{notif_det_text}' => $notif_det_text,
                    '{notif_det_date}' => $notif_det_date,
                    '{firstname}' => $customer->firstname,
                    '{lastname}' => $customer->lastname,
                    '{email}' => $customer->email,
                    '{delivery_block_txt}' => ConfigHistory::getFormatedAddress($delivery, '\n'),
                    '{invoice_block_txt}' => ConfigHistory::getFormatedAddress($invoice, '\n'),
                    '{delivery_block_html}' => ConfigHistory::getFormatedAddress(
                        $delivery, '<br />', [
                            'firstname' => '%s',
                            'lastname' => '%s',
                        ]
                    ),
                    '{invoice_block_html}' => ConfigHistory::getFormatedAddress(
                        $invoice, '<br />', [
                            'firstname' => '%s',
                            'lastname' => '%s',
                        ]
                    ),
                    '{delivery_company}' => $delivery->company,
                    '{delivery_firstname}' => $delivery->firstname,
                    '{delivery_lastname}' => $delivery->lastname,
                    '{delivery_address1}' => $delivery->address1,
                    '{delivery_address2}' => $delivery->address2,
                    '{delivery_city}' => $delivery->city,
                    '{delivery_postal_code}' => $delivery->postcode,
                    '{delivery_country}' => $delivery->country,
                    '{delivery_state}' => $delivery->id_state ? $delivery_state->name : '',
                    '{delivery_phone}' => $delivery->phone ? $delivery->phone : $delivery->phone_mobile,
                    '{delivery_other}' => $delivery->other,
                    '{invoice_company}' => $invoice->company,
                    '{invoice_firstname}' => $invoice->firstname,
                    '{invoice_lastname}' => $invoice->lastname,
                    '{invoice_address2}' => $invoice->address2,
                    '{invoice_address1}' => $invoice->address1,
                    '{invoice_city}' => $invoice->city,
                    '{invoice_postal_code}' => $invoice->postcode,
                    '{invoice_country}' => $invoice->country,
                    '{invoice_state}' => $invoice->id_state ? $invoice_state->name : '',
                    '{invoice_phone}' => $invoice->phone ? $invoice->phone : $invoice->phone_mobile,
                    '{invoice_other}' => $invoice->other,
                    '{order_name}' => $order->reference,
                    '{order_status}' => $order_state->name[$this->context->language->id],
                    '{shop_name}' => $configuration['PS_SHOP_NAME'],
                    '{carrier}' => (($carrier->name == '0') ? $configuration['PS_SHOP_NAME'] : $carrier->name),
                    '{payment}' => Tools::substr($order->payment, 0, 32),
                    '{total_paid}' => Context::getContext()->getCurrentLocale()->formatPrice(
                        (float) $order->total_paid,
                        $currency->iso_code
                    ),
                    '{total_products}' => Context::getContext()->getCurrentLocale()->formatPrice(
                        (float) $total_products,
                        $currency->iso_code
                    ),
                    '{total_discounts}' => Context::getContext()->getCurrentLocale()->formatPrice(
                        (float) $order->total_discounts,
                        $currency->iso_code
                    ),
                    '{total_shipping}' => Context::getContext()->getCurrentLocale()->formatPrice(
                        (float) $order->total_shipping,
                        $currency->iso_code
                    ),
                    '{total_tax_paid}' => Context::getContext()->getCurrentLocale()->formatPrice(
                        (float) ($order->total_products_wt - $order->total_products) + ($order->total_shipping_tax_incl - $order->total_shipping_tax_excl),
                        $currency->iso_code
                    ),
                    '{total_wrapping}' => Context::getContext()->getCurrentLocale()->formatPrice(
                        (float) $order->total_wrapping,
                        $currency->iso_code
                    ),
                    '{currency}' => $currency->sign,
                    '{gift}' => (bool) $order->gift,
                    '{gift_message}' => $order->gift_message,
                    '{message}' => $message,
                ],
                pSQL($to),
                null,
                null,
                null,
                null,
                null,
                dirname(__FILE__) . '/mails/',
                false,
                $this->context->shop->id
            );
        }
    }

    /**
     *  From ps_emailalert  module
     */
    public function getAllMessages($id)
    {
        $messages = Db::getInstance()->executeS('
            SELECT `message`
            FROM `' . _DB_PREFIX_ . 'message`
            WHERE `id_order` = ' . (int) $id . '
            ORDER BY `id_message` ASC');
        $result = [];
        foreach ($messages as $message) {
            $result[] = $message['message'];
        }

        return implode('<br/>', $result);
    }

    public function getPaymentMethods()
    {
        $payment_methods = [];
        foreach (PaymentModule::getInstalledPaymentModules() as $payment) {
            $module = Module::getInstanceByName($payment['name']);
            if (Validate::isLoadedObject($module) && $module->active) {
                $payment_methods[$module->id] = $module->displayName;
            }
        }

        return $payment_methods;
    }

    public function addNewMatches($id_carrier, $livraison_id)
    {
        $message = '';

        if (Tools::isSubmit('submitCarrierMatches')) {
            $CarrierMatches = new CarrierMatches();
            $CarrierMatches->id_carrier = $id_carrier;
            $CarrierMatches->livraison_id = $livraison_id;
            $CarrierMatches->id_shop = (int) Tools::getValue('id_shop');

            $CarrierMatches->add();

            $message = $this->displayConfirmation($this->l('Carrier Added Successfully ! '));
        }

        return $message;
    }

    public function addNewPaymentMatches($id_payment, $cash_payment_id)
    {
        $message = '';

        if (Tools::isSubmit('submitPaymentMatches')) {
            $PaymentMatches = new PaymentMatches();
            $PaymentMatches->id_payment = $id_payment;
            $PaymentMatches->cash_payment_id = $cash_payment_id;
            $PaymentMatches->id_shop = (int) Tools::getValue('id_shop');

            $PaymentMatches->add();

            $message = $this->displayConfirmation($this->l('Payment  Added Successfully !'));
        }

        return $message;
    }
}
